bl_info = {
    "name": "DXS Blender Render Monitoring",
    "author": "Dimensional X Studios",
    "version": (1, 3, 2),
    "blender": (5, 0, 0),
    "location": "View3D > Sidebar > DXS > Render Monitoring",
    "description": "Sync Blender 5.0+ render telemetry with DXS Internal App.",
    "category": "System",
}

import json
import os
import platform
import subprocess
import sys
import time
import traceback
import urllib.error
import urllib.parse
import urllib.request
import uuid
from datetime import datetime, timezone

import bpy
from bpy.app.handlers import persistent
from bpy.props import BoolProperty, FloatProperty, StringProperty

from . import config as addon_config


ADDON_VERSION = "1.3.2"
MIN_BLENDER_VERSION = (5, 0, 0)
SUPPORTED_VERSION = bpy.app.version >= MIN_BLENDER_VERSION

STATE = {
    "running": False,
    "job_id": "",
    "project_id": "",
    "last_frame_timestamp": 0.0,
    "frame_times": [],
    "last_progress": 0.0,
    "last_error": "",
    "timers_enabled": False,
}


def _prefs():
    return bpy.context.preferences.addons[__name__].preferences


def _now_iso():
    return datetime.now(timezone.utc).isoformat()


def _debug(message):
    print(f"[DXS Render Monitoring] {message}")


def _safe_float(value, fallback=0.0):
    try:
        return float(value)
    except Exception:
        return float(fallback)


def _safe_int(value, fallback=0):
    try:
        return int(value)
    except Exception:
        return int(fallback)


def _scene():
    return bpy.context.scene if bpy.context else None


def _abs_path(path):
    if not path:
        return ""
    try:
        return bpy.path.abspath(path)
    except Exception:
        return path


def _supabase_url():
    return addon_config.SUPABASE_URL.strip().rstrip("/")


def _supabase_anon_key():
    return addon_config.SUPABASE_ANON_KEY.strip()


def _auth_base():
    return f"{_supabase_url()}/auth/v1"


def _rest_base():
    return f"{_supabase_url()}/rest/v1"


def _functions_base():
    return f"{_supabase_url()}/functions/v1"


def _json_request(url, method="GET", headers=None, payload=None, timeout=25):
    data = None
    if payload is not None:
        data = json.dumps(payload).encode("utf-8")
    request = urllib.request.Request(url, data=data, headers=headers or {}, method=method)
    try:
        with urllib.request.urlopen(request, timeout=timeout) as response:
            raw = response.read().decode("utf-8")
            if not raw:
                return None
            return json.loads(raw)
    except urllib.error.HTTPError as error:
        try:
            detail = error.read().decode("utf-8")
        except Exception:
            detail = str(error)
        raise RuntimeError(f"HTTP {error.code}: {detail}") from error
    except urllib.error.URLError as error:
        raise RuntimeError(f"Network error: {error}") from error


def _auth_headers(token=None):
    headers = {
        "apikey": _supabase_anon_key(),
        "Content-Type": "application/json",
    }
    if token:
        headers["Authorization"] = f"Bearer {token}"
    return headers


def _rest_request(path, method="GET", token=None, payload=None, params=None, prefer=None):
    if not _supabase_url() or not _supabase_anon_key():
        raise RuntimeError("Supabase config is missing.")
    query = ""
    if params:
        clean = {key: value for key, value in params.items() if value is not None}
        if clean:
            query = f"?{urllib.parse.urlencode(clean)}"
    url = f"{_rest_base()}{path}{query}"
    headers = _auth_headers(token=token)
    if prefer:
        headers["Prefer"] = prefer
    return _json_request(url, method=method, headers=headers, payload=payload)


def _function_request(name, token=None, payload=None):
    if not _supabase_url() or not _supabase_anon_key():
        raise RuntimeError("Supabase config is missing.")
    url = f"{_functions_base()}/{name}"
    headers = _auth_headers(token=token)
    return _json_request(url, method="POST", headers=headers, payload=payload)


def _token_valid():
    settings = _prefs()
    return bool(settings.access_token) and time.time() < settings.token_expires_at


def _save_tokens(payload):
    settings = _prefs()
    settings.access_token = payload.get("access_token", "")
    settings.refresh_token = payload.get("refresh_token", "")
    expires_in = _safe_int(payload.get("expires_in"), 3600)
    settings.token_expires_at = time.time() + max(60, expires_in - 30)


def _fetch_user():
    settings = _prefs()
    if not settings.access_token:
        return False
    data = _json_request(
        f"{_auth_base()}/user",
        method="GET",
        headers=_auth_headers(token=settings.access_token),
    )
    if not data:
        return False
    settings.user_id = data.get("id", "")
    settings.user_email = data.get("email", settings.email)
    return bool(settings.user_id)


def _sign_in():
    settings = _prefs()
    if not settings.email or not settings.password:
        raise RuntimeError("Email and password are required.")
    payload = {"email": settings.email, "password": settings.password}
    data = _json_request(
        f"{_auth_base()}/token?grant_type=password",
        method="POST",
        headers=_auth_headers(),
        payload=payload,
    )
    _save_tokens(data or {})
    if not _fetch_user():
        raise RuntimeError("Could not fetch profile after sign-in.")
    _upsert_machine()
    return True


def _refresh_session():
    settings = _prefs()
    if not settings.refresh_token:
        return False
    payload = {"refresh_token": settings.refresh_token}
    data = _json_request(
        f"{_auth_base()}/token?grant_type=refresh_token",
        method="POST",
        headers=_auth_headers(),
        payload=payload,
    )
    _save_tokens(data or {})
    if not settings.user_id:
        _fetch_user()
    return bool(settings.access_token)


def ensure_access_token(allow_login=True):
    if not SUPPORTED_VERSION:
        return ""
    settings = _prefs()
    if _token_valid():
        if not settings.user_id:
            _fetch_user()
        return settings.access_token
    if settings.refresh_token:
        try:
            if _refresh_session():
                return settings.access_token
        except Exception as error:
            _debug(f"Refresh failed: {error}")
    if allow_login and settings.email and settings.password:
        try:
            _sign_in()
            return settings.access_token
        except Exception as error:
            settings.last_error = str(error)
            _debug(f"Sign-in failed: {error}")
    return ""


def _ensure_machine_id():
    settings = _prefs()
    if not settings.machine_id:
        settings.machine_id = str(uuid.uuid4())
    return settings.machine_id


def _scene_view_layers(scene):
    names = []
    active_name = ""
    try:
        names = [layer.name for layer in scene.view_layers]
        active = scene.view_layers.active
        active_name = active.name if active else ""
    except Exception:
        names = []
        active_name = ""
    return active_name, names


def _collect_color(scene):
    view_transform = ""
    look = ""
    try:
        view_transform = scene.view_settings.view_transform or ""
        look = scene.view_settings.look or ""
    except Exception:
        pass
    return view_transform, look


def _collect_output(scene):
    image_settings = scene.render.image_settings
    file_format = getattr(image_settings, "file_format", "") or ""
    compression = ""
    exr_codec = ""
    color_depth = ""
    if hasattr(image_settings, "compression"):
        compression = str(getattr(image_settings, "compression", ""))
    if hasattr(image_settings, "quality"):
        quality = str(getattr(image_settings, "quality", ""))
        compression = f"{compression} q{quality}".strip()
    if file_format.startswith("OPEN_EXR"):
        if hasattr(image_settings, "exr_codec"):
            exr_codec = str(getattr(image_settings, "exr_codec", ""))
            compression = exr_codec or compression
        if hasattr(image_settings, "color_depth"):
            color_depth = str(getattr(image_settings, "color_depth", ""))
    return {
        "output_format": file_format,
        "compression": compression,
        "exr_codec": exr_codec,
        "color_depth": color_depth,
    }


def _collect_cycles(scene):
    device = ""
    samples = 0
    if scene.render.engine == "CYCLES":
        cycles = getattr(scene, "cycles", None)
        if cycles:
            device = str(getattr(cycles, "device", "") or "")
            samples = _safe_int(getattr(cycles, "samples", 0), 0)
    return device, samples


def _render_fps(scene):
    fps_base = _safe_float(scene.render.fps_base, 1.0)
    if fps_base == 0:
        fps_base = 1.0
    fps = _safe_float(scene.render.fps, 24.0) / fps_base
    return fps, fps_base


def _project_id(scene):
    existing = scene.get("dxs_project_id")
    if existing:
        return str(existing)
    settings = _prefs()
    blend_path = bpy.data.filepath or scene.name
    source = f"{settings.user_id}:{blend_path}"
    value = str(uuid.uuid5(uuid.NAMESPACE_URL, source))
    scene["dxs_project_id"] = value
    return value


def _collect_project_payload(scene, mark_rendered=False):
    settings = _prefs()
    fps, fps_base = _render_fps(scene)
    resolution_x = int(scene.render.resolution_x * (scene.render.resolution_percentage / 100.0))
    resolution_y = int(scene.render.resolution_y * (scene.render.resolution_percentage / 100.0))
    device, samples = _collect_cycles(scene)
    output = _collect_output(scene)
    view_layer_name, view_layers = _scene_view_layers(scene)
    color_view, color_look = _collect_color(scene)
    payload = {
        "id": _project_id(scene),
        "user_id": settings.user_id,
        "machine_id": _ensure_machine_id(),
        "name": os.path.basename(bpy.data.filepath) if bpy.data.filepath else "Untitled",
        "blend_path": bpy.data.filepath or "",
        "scene_name": scene.name,
        "frame_start": int(scene.frame_start),
        "frame_end": int(scene.frame_end),
        "fps": fps,
        "fps_base": fps_base,
        "resolution_x": resolution_x,
        "resolution_y": resolution_y,
        "render_engine": scene.render.engine,
        "device": device,
        "samples": samples,
        "output_path": _abs_path(scene.render.filepath),
        "output_format": output["output_format"],
        "compression": output["compression"],
        "exr_codec": output["exr_codec"],
        "color_depth": output["color_depth"],
        "color_view_transform": color_view,
        "color_look": color_look,
        "view_layer_name": view_layer_name,
        "view_layers": view_layers,
        "last_opened_at": _now_iso(),
        "updated_at": _now_iso(),
    }
    if mark_rendered:
        payload["last_rendered_at"] = _now_iso()
    return payload


def _upsert_machine():
    token = ensure_access_token(allow_login=False)
    settings = _prefs()
    if not token or not settings.user_id:
        return False
    payload = {
        "machine_id": _ensure_machine_id(),
        "user_id": settings.user_id,
        "hostname": platform.node() or "Unknown",
        "os": platform.platform(),
        "blender_version": bpy.app.version_string,
        "addon_version": ADDON_VERSION,
        "last_seen_at": _now_iso(),
        "updated_at": _now_iso(),
        "capabilities": {
            "python": sys.version.split(" ")[0],
            "render_engines": list(getattr(bpy.app, "render_engines", [])),
        },
    }
    _rest_request(
        "/blender_machines",
        method="POST",
        token=token,
        payload=[payload],
        params={"on_conflict": "machine_id"},
        prefer="resolution=merge-duplicates,return=minimal",
    )
    return True


def sync_project(mark_rendered=False):
    scene = _scene()
    if scene is None:
        return False
    token = ensure_access_token()
    settings = _prefs()
    if not token or not settings.user_id:
        return False
    _upsert_machine()
    payload = _collect_project_payload(scene, mark_rendered=mark_rendered)
    _rest_request(
        "/blender_projects",
        method="POST",
        token=token,
        payload=[payload],
        params={"on_conflict": "id"},
        prefer="resolution=merge-duplicates,return=minimal",
    )
    settings.last_sync_at = _now_iso()
    STATE["project_id"] = payload["id"]
    return True


def _insert_event(kind, message, payload=None):
    token = ensure_access_token(allow_login=False)
    settings = _prefs()
    if not token or not settings.user_id or not STATE["job_id"]:
        return
    data = {
        "job_id": STATE["job_id"],
        "user_id": settings.user_id,
        "kind": kind,
        "message": message[:8000],
        "payload": payload or {},
    }
    try:
        _rest_request("/blender_render_events", method="POST", token=token, payload=[data])
    except Exception as error:
        _debug(f"Event insert failed: {error}")


def _insert_notification(status, title, body):
    token = ensure_access_token(allow_login=False)
    settings = _prefs()
    if not token or not settings.user_id or not STATE["job_id"]:
        return
    owner_name = _fetch_profile_name(token, settings.user_id) or settings.user_email or settings.email or "A user"
    render_label = _current_render_label()
    admin_ids = _fetch_admin_producer_ids(token)
    can_broadcast = settings.user_id in set(admin_ids)
    recipients = {settings.user_id}
    if can_broadcast:
        for admin_id in admin_ids:
            recipients.add(admin_id)
    payload = []
    normalized = str(status or "").lower()
    for recipient_id in recipients:
        if recipient_id == settings.user_id:
            notification_title = title
            notification_body = body
        elif normalized == "completed":
            notification_title = "Render finished in studio"
            notification_body = f"{owner_name}'s render finished successfully ({render_label})."
        elif normalized == "failed":
            notification_title = "Render failed in studio"
            notification_body = f"{owner_name}'s render failed ({render_label})."
        elif normalized == "crashed":
            notification_title = "Render crashed in studio"
            notification_body = f"{owner_name}'s render crashed ({render_label})."
        elif normalized == "canceled":
            notification_title = "Render canceled in studio"
            notification_body = f"{owner_name}'s render was canceled ({render_label})."
        else:
            notification_title = "Render update"
            notification_body = f"{owner_name}'s render updated ({render_label})."
        payload.append(
            {
                "user_id": recipient_id,
                "job_id": STATE["job_id"],
                "type": status,
                "title": notification_title,
                "body": notification_body,
                "read": False,
            }
        )
    if not payload:
        return
    try:
        _rest_request("/blender_notifications", method="POST", token=token, payload=payload)
    except Exception as error:
        _debug(f"Notification insert failed: {error}")


def _fetch_profile_name(token, user_id):
    if not token or not user_id:
        return ""
    try:
        rows = _rest_request(
            "/profiles",
            method="GET",
            token=token,
            params={
                "select": "name",
                "id": f"eq.{user_id}",
                "limit": 1,
            },
        )
        if isinstance(rows, list) and rows:
            return str(rows[0].get("name") or "").strip()
    except Exception as error:
        _debug(f"Profile name lookup failed: {error}")
    return ""


def _fetch_admin_producer_ids(token):
    if not token:
        return []
    try:
        rows = _rest_request(
            "/profiles",
            method="GET",
            token=token,
            params={
                "select": "id",
                "role": "in.(admin,producer)",
                "limit": 200,
            },
        )
        if isinstance(rows, list):
            return [str(item.get("id") or "").strip() for item in rows if item.get("id")]
    except Exception as error:
        _debug(f"Recipient lookup failed: {error}")
    return []


def _current_render_label():
    if bpy.data.filepath:
        return os.path.basename(bpy.data.filepath)
    scene = _scene()
    return scene.name if scene else "Untitled render"


def _send_render_finish_push(status):
    token = ensure_access_token(allow_login=False)
    settings = _prefs()
    if not token or not settings.user_id:
        return
    recipients = {settings.user_id}
    for admin_id in _fetch_admin_producer_ids(token):
        recipients.add(admin_id)
    recipient_ids = [user_id for user_id in recipients if user_id]
    if not recipient_ids:
        return

    owner_name = _fetch_profile_name(token, settings.user_id) or settings.user_email or settings.email or "A user"
    render_label = _current_render_label()
    normalized = str(status or "").lower()
    if normalized == "completed":
        title = "Render finished"
        body = f"{owner_name}'s render finished successfully ({render_label})."
    elif normalized == "failed":
        title = "Render failed"
        body = f"{owner_name}'s render failed ({render_label})."
    elif normalized == "crashed":
        title = "Render crashed"
        body = f"{owner_name}'s render crashed ({render_label})."
    else:
        title = "Render update"
        body = f"{owner_name}'s render updated ({render_label})."

    payload = {
        "access_token": token,
        "title": title,
        "body": body,
        "user_ids": recipient_ids,
    }
    try:
        _function_request("push-notify", token=token, payload=payload)
    except Exception as error:
        _debug(f"Push send failed: {error}")


def _update_job(payload):
    token = ensure_access_token(allow_login=False)
    if not token or not STATE["job_id"]:
        return False
    payload["updated_at"] = _now_iso()
    _rest_request(
        "/blender_render_jobs",
        method="PATCH",
        token=token,
        payload=payload,
        params={"id": f"eq.{STATE['job_id']}"},
        prefer="return=minimal",
    )
    return True


def _create_job(scene):
    settings = _prefs()
    token = ensure_access_token()
    if not token or not settings.user_id:
        return ""
    if not sync_project(mark_rendered=False):
        return ""
    _upsert_machine()

    fps, _fps_base = _render_fps(scene)
    resolution_x = int(scene.render.resolution_x * (scene.render.resolution_percentage / 100.0))
    resolution_y = int(scene.render.resolution_y * (scene.render.resolution_percentage / 100.0))
    output = _collect_output(scene)
    device, samples = _collect_cycles(scene)
    view_layer_name, view_layers = _scene_view_layers(scene)
    color_view, color_look = _collect_color(scene)

    frame_start = int(scene.frame_start)
    frame_end = int(scene.frame_end)
    total_frames = max(0, frame_end - frame_start + 1)
    payload = {
        "user_id": settings.user_id,
        "project_id": STATE["project_id"] or _project_id(scene),
        "machine_id": _ensure_machine_id(),
        "status": "running",
        "progress": 0.0,
        "current_frame": frame_start,
        "frame_start": frame_start,
        "frame_end": frame_end,
        "total_frames": total_frames,
        "fps": fps,
        "avg_frame_time_s": None,
        "last_frame_time_s": None,
        "eta_s": None,
        "started_at": _now_iso(),
        "last_heartbeat_at": _now_iso(),
        "output_path": _abs_path(scene.render.filepath),
        "output_format": output["output_format"],
        "compression": output["compression"],
        "render_engine": scene.render.engine,
        "device": device,
        "samples": samples,
        "resolution_x": resolution_x,
        "resolution_y": resolution_y,
        "color_view_transform": color_view,
        "color_look": color_look,
        "scene_name": scene.name,
        "view_layer_name": view_layer_name,
        "view_layers": view_layers,
        "last_error": None,
    }
    rows = _rest_request(
        "/blender_render_jobs",
        method="POST",
        token=token,
        payload=[payload],
        params={"select": "id"},
        prefer="return=representation",
    )
    if not rows:
        return ""
    return rows[0].get("id", "")


def _finish_job(status, error_message=""):
    if not STATE["job_id"]:
        return
    payload = {
        "status": status,
        "finished_at": _now_iso(),
        "last_heartbeat_at": _now_iso(),
    }
    if status == "completed":
        payload["progress"] = 1.0
    if error_message:
        payload["last_error"] = error_message[:8000]
    _update_job(payload)
    settings = _prefs()
    settings.last_render_at = _now_iso()

    if status == "completed":
        sync_project(mark_rendered=True)
        _insert_notification("completed", "Render completed", "Your Blender render finished successfully.")
        _send_render_finish_push("completed")
    elif status == "failed":
        _insert_notification("failed", "Render failed", error_message or "Render failed.")
        _send_render_finish_push("failed")
    elif status == "crashed":
        _insert_notification("crashed", "Render crashed", error_message or "Render crashed.")
        _send_render_finish_push("crashed")

    STATE["running"] = False
    STATE["job_id"] = ""
    STATE["last_frame_timestamp"] = 0.0
    STATE["frame_times"] = []
    STATE["last_progress"] = 0.0
    STATE["last_error"] = error_message or ""


def _current_job_status():
    if STATE["running"] and STATE["job_id"]:
        return "running"
    return "idle"


def _frame_update(scene):
    if not STATE["job_id"]:
        return
    timestamp = time.time()
    frame_time = None
    if STATE["last_frame_timestamp"] > 0:
        frame_time = max(0.0001, timestamp - STATE["last_frame_timestamp"])
        STATE["frame_times"].append(frame_time)
        STATE["frame_times"] = STATE["frame_times"][-120:]
    STATE["last_frame_timestamp"] = timestamp

    frame_start = _safe_int(scene.frame_start, 0)
    frame_end = _safe_int(scene.frame_end, 0)
    current_frame = _safe_int(scene.frame_current, frame_start)
    total_frames = max(0, _safe_int(frame_end - frame_start + 1, 0))
    frame_done = max(0, current_frame - frame_start + 1)
    progress = (float(frame_done) / float(total_frames)) if total_frames else 0.0
    progress = max(0.0, min(1.0, progress))
    STATE["last_progress"] = progress

    avg_frame = 0.0
    if STATE["frame_times"]:
        avg_frame = sum(STATE["frame_times"]) / len(STATE["frame_times"])
    remaining_frames = max(0, frame_end - current_frame)
    eta_s = int(avg_frame * remaining_frames) if avg_frame > 0 else None

    payload = {
        "current_frame": current_frame,
        "progress": progress,
        "last_frame_time_s": frame_time,
        "avg_frame_time_s": avg_frame if avg_frame > 0 else None,
        "eta_s": eta_s,
        "last_heartbeat_at": _now_iso(),
    }
    _update_job(payload)


@persistent
def handle_load_post(_dummy):
    if not SUPPORTED_VERSION:
        return
    try:
        sync_project(mark_rendered=False)
    except Exception as error:
        _prefs().last_error = str(error)
        _debug(f"load_post sync failed: {error}")


@persistent
def handle_save_post(_dummy):
    if not SUPPORTED_VERSION:
        return
    try:
        sync_project(mark_rendered=False)
    except Exception as error:
        _prefs().last_error = str(error)
        _debug(f"save_post sync failed: {error}")


@persistent
def handle_render_pre(scene, _depsgraph):
    if not SUPPORTED_VERSION:
        return
    # Blender can fire render_pre for every frame during animation renders.
    # Guard to keep a single job row for the full render session.
    if STATE["running"] and STATE["job_id"]:
        return
    try:
        token = ensure_access_token()
        if not token:
            return
        STATE["job_id"] = _create_job(scene)
        STATE["running"] = bool(STATE["job_id"])
        STATE["last_frame_timestamp"] = time.time()
        STATE["frame_times"] = []
        STATE["last_progress"] = 0.0
        STATE["last_error"] = ""
    except Exception as error:
        STATE["running"] = False
        STATE["job_id"] = ""
        _prefs().last_error = str(error)
        _debug(f"render_pre failed: {error}")
        _debug(traceback.format_exc())


@persistent
def handle_render_write(scene, _depsgraph):
    if not SUPPORTED_VERSION or not STATE["job_id"]:
        return
    try:
        _frame_update(scene)
    except Exception as error:
        message = str(error)
        _prefs().last_error = message
        _insert_event("error", "render_write failed", {"error": message})
        _finish_job("failed", message)
        _debug(f"render_write failed: {error}")
        _debug(traceback.format_exc())


@persistent
def handle_render_post(_scene, _depsgraph):
    # render_post can be called for every frame; do not finish the job here.
    # Completion is handled by render_complete.
    return


@persistent
def handle_render_complete(_scene):
    if not SUPPORTED_VERSION or not STATE["job_id"]:
        return
    try:
        _finish_job("completed", "")
    except Exception as error:
        message = str(error)
        _prefs().last_error = message
        _insert_event("error", "render_complete failed", {"error": message})
        _finish_job("failed", message)


@persistent
def handle_render_cancel(_scene, _depsgraph):
    if not SUPPORTED_VERSION:
        return
    try:
        _finish_job("canceled", "")
    except Exception as error:
        message = str(error)
        _prefs().last_error = message
        _insert_event("error", "render_cancel failed", {"error": message})


def _parse_command_payload(command):
    payload = command.get("payload")
    if isinstance(payload, dict):
        return payload
    if isinstance(payload, str):
        try:
            parsed = json.loads(payload)
            if isinstance(parsed, dict):
                return parsed
        except Exception:
            return {}
    return {}


def _mark_command(command_id, status, result):
    token = ensure_access_token(allow_login=False)
    if not token or not command_id:
        return
    payload = {
        "status": status,
        "result": result[:8000] if result else "",
        "updated_at": _now_iso(),
    }
    _rest_request(
        "/blender_render_commands",
        method="PATCH",
        token=token,
        payload=payload,
        params={"id": f"eq.{command_id}"},
        prefer="return=minimal",
    )


def _reveal_output_path(path):
    if not path:
        raise RuntimeError("No output path in command payload.")
    target = path
    if not os.path.isdir(target):
        target = os.path.dirname(target)
    if not target:
        raise RuntimeError("Could not resolve output folder.")
    if os.name == "nt":
        os.startfile(target)  # type: ignore[attr-defined]
        return
    if sys.platform == "darwin":
        subprocess.Popen(["open", target])
        return
    subprocess.Popen(["xdg-open", target])


def _run_command(command):
    prefs = _prefs()
    command_id = command.get("id", "")
    command_type = command.get("type", "")
    payload = _parse_command_payload(command)

    _mark_command(command_id, "processing", "Processing command.")
    try:
        if command_type == "cancel":
            if STATE["running"]:
                try:
                    bpy.ops.render.cancel("INVOKE_DEFAULT")
                except Exception:
                    bpy.ops.render.cancel()
                result = "Cancel requested."
            else:
                result = "Render already idle."
            _mark_command(command_id, "done", result)
            return

        if command_type == "resume":
            if STATE["running"]:
                _mark_command(command_id, "done", "Render already running.")
                return
            blend_path = payload.get("blend_path") or payload.get("path")
            if blend_path and bpy.data.filepath and os.path.normpath(bpy.data.filepath) != os.path.normpath(blend_path):
                if prefs.allow_remote_open_project:
                    bpy.ops.wm.open_mainfile(filepath=blend_path)
                else:
                    _mark_command(command_id, "failed", "Remote project switch blocked by settings.")
                    return
            animation = bool(payload.get("animation", True))
            try:
                bpy.ops.render.render("INVOKE_DEFAULT", animation=animation)
            except Exception:
                bpy.ops.render.render(animation=animation)
            _mark_command(command_id, "done", "Render resume requested.")
            return

        if command_type == "open_project":
            if not prefs.allow_remote_open_project:
                raise RuntimeError("Remote open project is disabled in add-on settings.")
            blend_path = payload.get("blend_path") or payload.get("path")
            if not blend_path:
                raise RuntimeError("Missing blend_path in command payload.")
            bpy.ops.wm.open_mainfile(filepath=blend_path)
            _mark_command(command_id, "done", "Project opened.")
            return

        if command_type == "reveal_output":
            if not prefs.allow_remote_reveal_output:
                raise RuntimeError("Remote reveal output is disabled in add-on settings.")
            output_path = payload.get("output_path") or payload.get("path")
            _reveal_output_path(output_path)
            _mark_command(command_id, "done", "Output location opened.")
            return

        raise RuntimeError(f"Unsupported command type: {command_type}")
    except Exception as error:
        _mark_command(command_id, "failed", str(error))
        _debug(f"Command failed: {error}")


def heartbeat_timer():
    if not STATE["timers_enabled"]:
        return None
    interval = max(5.0, _safe_float(_prefs().heartbeat_interval, 15.0))
    if not SUPPORTED_VERSION:
        return interval
    try:
        token = ensure_access_token(allow_login=False)
        if token:
            _upsert_machine()
            if STATE["job_id"]:
                _update_job({"last_heartbeat_at": _now_iso()})
    except Exception as error:
        _prefs().last_error = str(error)
        _debug(f"Heartbeat failed: {error}")
    return interval


def command_poll_timer():
    if not STATE["timers_enabled"]:
        return None
    interval = min(10.0, max(1.0, _safe_float(_prefs().command_poll_interval, 2.0)))
    if not SUPPORTED_VERSION:
        return interval
    token = ensure_access_token(allow_login=False)
    settings = _prefs()
    if not token or not settings.machine_id:
        return interval
    try:
        commands = _rest_request(
            "/blender_render_commands",
            method="GET",
            token=token,
            params={
                "select": "id,type,payload,status,job_id,machine_id",
                "machine_id": f"eq.{settings.machine_id}",
                "status": "eq.queued",
                "order": "created_at.asc",
                "limit": 10,
            },
        )
        if isinstance(commands, list):
            for command in commands:
                _run_command(command)
    except Exception as error:
        settings.last_error = str(error)
        _debug(f"Command polling failed: {error}")
    return interval


class DXSRenderMonitoringPreferences(bpy.types.AddonPreferences):
    bl_idname = __name__

    email: StringProperty(name="Email")
    password: StringProperty(name="Password", subtype="PASSWORD")
    user_id: StringProperty(name="User ID", options={"HIDDEN"})
    user_email: StringProperty(name="User Email", options={"HIDDEN"})
    access_token: StringProperty(name="Access Token", options={"HIDDEN"})
    refresh_token: StringProperty(name="Refresh Token", options={"HIDDEN"})
    token_expires_at: FloatProperty(name="Token Expires", options={"HIDDEN"})
    machine_id: StringProperty(name="Machine ID", options={"HIDDEN"})
    last_sync_at: StringProperty(name="Last Sync", options={"HIDDEN"})
    last_render_at: StringProperty(name="Last Render", options={"HIDDEN"})
    last_error: StringProperty(name="Last Error", options={"HIDDEN"})
    heartbeat_interval: FloatProperty(
        name="Heartbeat Interval (s)",
        default=15.0,
        min=5.0,
        max=120.0,
    )
    command_poll_interval: FloatProperty(
        name="Command Poll Interval (s)",
        default=2.0,
        min=1.0,
        max=10.0,
    )
    allow_remote_open_project: BoolProperty(
        name="Allow remote open project",
        default=False,
    )
    allow_remote_reveal_output: BoolProperty(
        name="Allow remote reveal output",
        default=False,
    )

    def draw(self, _context):
        layout = self.layout
        if not SUPPORTED_VERSION:
            layout.label(
                text="Blender 5.0+ is required. Add-on features are disabled.",
                icon="ERROR",
            )
            return
        layout.label(text="DXS account")
        layout.prop(self, "email")
        layout.prop(self, "password")
        layout.separator()
        layout.label(text=f"Machine ID: {self.machine_id or 'Not generated yet'}")
        layout.prop(self, "heartbeat_interval")
        layout.prop(self, "command_poll_interval")
        layout.separator()
        layout.prop(self, "allow_remote_open_project")
        layout.prop(self, "allow_remote_reveal_output")
        if self.last_error:
            layout.separator()
            layout.label(text=f"Last error: {self.last_error}", icon="ERROR")


class DXS_RENDERMON_OT_sign_in(bpy.types.Operator):
    bl_idname = "dxs_render_monitor.sign_in"
    bl_label = "Sign In"

    def execute(self, _context):
        if not SUPPORTED_VERSION:
            self.report({"ERROR"}, "Blender 5.0+ is required.")
            return {"CANCELLED"}
        try:
            _sign_in()
            self.report({"INFO"}, "Signed in successfully.")
            return {"FINISHED"}
        except Exception as error:
            _prefs().last_error = str(error)
            self.report({"ERROR"}, str(error))
            return {"CANCELLED"}


class DXS_RENDERMON_OT_sign_out(bpy.types.Operator):
    bl_idname = "dxs_render_monitor.sign_out"
    bl_label = "Sign Out"

    def execute(self, _context):
        settings = _prefs()
        settings.access_token = ""
        settings.refresh_token = ""
        settings.token_expires_at = 0.0
        settings.user_id = ""
        settings.user_email = ""
        self.report({"INFO"}, "Signed out.")
        return {"FINISHED"}


class DXS_RENDERMON_OT_sync_project(bpy.types.Operator):
    bl_idname = "dxs_render_monitor.sync_project"
    bl_label = "Sync Project"

    def execute(self, _context):
        if not SUPPORTED_VERSION:
            self.report({"ERROR"}, "Blender 5.0+ is required.")
            return {"CANCELLED"}
        try:
            if sync_project(mark_rendered=False):
                self.report({"INFO"}, "Project synced.")
                return {"FINISHED"}
            self.report({"WARNING"}, "Could not sync project. Check account session.")
            return {"CANCELLED"}
        except Exception as error:
            _prefs().last_error = str(error)
            self.report({"ERROR"}, str(error))
            return {"CANCELLED"}


class DXS_RENDERMON_PT_panel(bpy.types.Panel):
    bl_label = "Render Monitoring"
    bl_idname = "DXS_RENDERMON_PT_panel"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = "DXS"

    def draw(self, _context):
        layout = self.layout
        settings = _prefs()

        if not SUPPORTED_VERSION:
            layout.label(
                text="Blender 5.0+ required. Features are disabled.",
                icon="ERROR",
            )
            return

        layout.label(text="Account", icon="USER")
        connected = bool(settings.access_token) and time.time() < settings.token_expires_at
        status_row = layout.row()
        status_row.label(
            text="Connected" if connected else "Not connected",
            icon="CHECKMARK" if connected else "CANCEL",
        )
        layout.prop(settings, "email")
        layout.prop(settings, "password")
        buttons = layout.row(align=True)
        buttons.operator("dxs_render_monitor.sign_in", icon="UNLOCKED")
        buttons.operator("dxs_render_monitor.sign_out", icon="LOCKED")

        layout.separator()
        layout.label(text="Project Sync", icon="FILE")
        layout.operator("dxs_render_monitor.sync_project", icon="FILE_REFRESH")
        layout.label(text=f"Machine ID: {settings.machine_id or 'Pending'}")
        if settings.last_sync_at:
            layout.label(text=f"Last sync: {settings.last_sync_at}")
        if settings.last_render_at:
            layout.label(text=f"Last render: {settings.last_render_at}")

        layout.separator()
        layout.label(text="Telemetry", icon="SEQUENCE")
        layout.label(text=f"Job status: {_current_job_status()}")
        if STATE["job_id"]:
            layout.label(text=f"Job ID: {STATE['job_id']}")
        if STATE["frame_times"]:
            last_frame = STATE["frame_times"][-1]
            avg_frame = sum(STATE["frame_times"]) / len(STATE["frame_times"])
            layout.label(text=f"Last frame: {last_frame:.2f}s")
            layout.label(text=f"Avg frame: {avg_frame:.2f}s")
        if STATE["last_progress"] > 0:
            layout.label(text=f"Progress: {STATE['last_progress'] * 100:.1f}%")

        layout.separator()
        layout.label(text="Remote Commands", icon="CONSOLE")
        layout.prop(settings, "allow_remote_open_project")
        layout.prop(settings, "allow_remote_reveal_output")
        layout.prop(settings, "heartbeat_interval")
        layout.prop(settings, "command_poll_interval")
        if settings.last_error:
            layout.separator()
            layout.label(text=f"Last error: {settings.last_error}", icon="ERROR")


CLASSES = (
    DXSRenderMonitoringPreferences,
    DXS_RENDERMON_OT_sign_in,
    DXS_RENDERMON_OT_sign_out,
    DXS_RENDERMON_OT_sync_project,
    DXS_RENDERMON_PT_panel,
)


def _register_handlers():
    mapping = (
        (bpy.app.handlers.load_post, handle_load_post),
        (bpy.app.handlers.save_post, handle_save_post),
        (bpy.app.handlers.render_pre, handle_render_pre),
        (bpy.app.handlers.render_write, handle_render_write),
        (bpy.app.handlers.render_post, handle_render_post),
        (bpy.app.handlers.render_complete, handle_render_complete),
        (bpy.app.handlers.render_cancel, handle_render_cancel),
    )
    for handler_list, handler in mapping:
        if handler not in handler_list:
            handler_list.append(handler)


def _unregister_handlers():
    mapping = (
        (bpy.app.handlers.load_post, handle_load_post),
        (bpy.app.handlers.save_post, handle_save_post),
        (bpy.app.handlers.render_pre, handle_render_pre),
        (bpy.app.handlers.render_write, handle_render_write),
        (bpy.app.handlers.render_post, handle_render_post),
        (bpy.app.handlers.render_complete, handle_render_complete),
        (bpy.app.handlers.render_cancel, handle_render_cancel),
    )
    for handler_list, handler in mapping:
        if handler in handler_list:
            handler_list.remove(handler)


def register():
    for cls in CLASSES:
        bpy.utils.register_class(cls)

    if not SUPPORTED_VERSION:
        _debug("Blender < 5.0 detected; add-on features disabled.")
        return

    STATE["timers_enabled"] = True
    _register_handlers()

    if not bpy.app.timers.is_registered(heartbeat_timer):
        bpy.app.timers.register(heartbeat_timer, first_interval=1.0, persistent=True)
    if not bpy.app.timers.is_registered(command_poll_timer):
        bpy.app.timers.register(command_poll_timer, first_interval=2.0, persistent=True)


def unregister():
    STATE["timers_enabled"] = False
    _unregister_handlers()

    for cls in reversed(CLASSES):
        bpy.utils.unregister_class(cls)
